<?php

    namespace Asn\Core;

    class Database
    {

        private $dbh;
        private $stmt;

        public function __construct(string $conn = 'db')
        {
            $settings = @include __DIR__ . '/../config/db.config.php';

            if (!is_array($settings) || !isset($settings[$conn])) {
                exit('Invalid db settings');
            }

            $dsn = 'mysql:host=' . $settings[$conn]['host'] . ';dbname=' . $settings[$conn]['dbname'] . ';charset=utf8';

            $options = [
                \PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
                \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC,
                \PDO::ATTR_EMULATE_PREPARES => false,
                \PDO::MYSQL_ATTR_FOUND_ROWS => true
            ];

            try {
                $this->dbh = new \PDO($dsn, $settings[$conn]['user'], $settings[$conn]['pass'], $options);
            } catch (\PDOException $e) {
                exit('Could not connect to the database');
            }
        }

        /**
         * Executes a query
         *
         * @param string $sql
         * @param array|null $params
         * @return \PDOStatement
         * @throws \Exception
         */
        public function exec(string $sql, array $params = null): \PDOStatement
        {
            if ($params) {
                try {
                    $this->stmt = $this->dbh->prepare($sql);

                    foreach ($params as $p) {
                        switch (true) {
                            case is_string($p[1]):
                                $p[2] = \PDO::PARAM_STR;
                                break;
                            case is_int($p[1]):
                                $p[2] = \PDO::PARAM_INT;
                                break;
                            case is_bool($p[1]):
                                $p[2] = \PDO::PARAM_BOOL;
                                break;
                            case is_null($p[1]):
                                $p[2] = \PDO::PARAM_NULL;
                                break;
                            default:
                                $p[2] = \PDO::PARAM_STR;
                        }

                        $this->stmt->bindValue($p[0], $p[1], $p[2]);
                    }

                    $this->stmt->execute();
                    return $this->stmt;
                } catch (\PDOException $e) {
                    throw $e;
                }
            } else {
                try {
                    return $this->dbh->query($sql);
                } catch (\PDOException $e) {
                    throw $e;
                }
            }
        }

        /**
         * Inserts multiple rows
         *
         * @param string $sql
         * @param array $data
         * @return \PDOStatement
         * @throws \Exception
         */
        public function insertMulti(string $sql = '', array $data = []): \PDOStatement
        {

            $rows_count = count($data);
            if ($rows_count === 0) {
                throw new \Exception('No data to insert');
            }

            $fields_str = '';
            $placeholders = '';

            $fields_arr = array_keys($data[0]);
            foreach ($fields_arr as $field) {
                $fields_str .= "$field,";
                $placeholders .= '?,';
            }

            $fields_str = rtrim($fields_str, ',');
            $row = '(' . rtrim($placeholders, ',') . '),';
            $rows = rtrim(str_repeat($row, $rows_count), ',');

            $values_arr = [];
            foreach ($data as $d) {
                $vals = array_values($d);
                foreach ($vals as $v) {
                    $values_arr[] = $v;
                }
            }

            $sql .= '(' . $fields_str . ') VALUES ' . $rows;
            try {
                $this->stmt = $this->dbh->prepare($sql);
                $this->stmt->execute($values_arr);
                return $this->stmt;
            } catch (\PDOException $e) {
                throw $e;
            }
        }

        /**
         * Returns last insert id
         *
         * @return string
         */
        public function lastInsertId(): string
        {
            return $this->dbh->lastInsertId();
        }

        /**
         * Starts a transaction
         *
         * @return bool
         * @throws \Exception
         */
        public function beginTransaction(): bool
        {
            try {
                return $this->dbh->beginTransaction();
            } catch (\PDOException $e) {
                throw $e;
            }
        }

        /**
         * Commits active transaction
         *
         * @return bool
         * @throws \Exception
         */
        public function commit(): bool
        {
            try {
                return $this->dbh->commit();
            } catch (\PDOException $e) {
                throw $e;
            }
        }

        /**
         * Rolls back active transaction
         *
         * @return bool
         * @throws \Exception
         */
        public function rollBack(): bool
        {
            try {
                return $this->dbh->rollBack();
            } catch (\PDOException $e) {
                throw $e;
            }
        }

        /**
         * Checks if transaction is active
         *
         * @return bool
         * @throws \Exception
         */
        public function inTransaction(): bool
        {
            try {
                return $this->dbh->inTransaction();
            } catch (\PDOException $e) {
                throw $e;
            }
        }

        /**
         * Returns PDO instance representing the connection
         *
         * @return \PDO
         */
        public function getConnection(): \PDO
        {
            return $this->dbh;
        }

    }