<?php

    namespace Asn\Core\Http;

    class Session {

        /**
         * Session settings
         *
         * @var array
         */
        private $options = [
            'name' => 'w',
            'referer_check' => '',
            'use_cookies' => 1,
            'use_only_cookies' => 1,
            'sue_trans_sid' => 0,
            'use_strict_mode' => 1,
            'cookie_lifetime' => 0,
            'cookie_path' => '/',
            'cookie_domain' => '', // cookie is available on current subdomain only
            'cookie_secure' => false, // available through HTTP and HTTPS
            'cookie_httponly' => true,
            'save_path' => '',
            'gc_probability' => 1,
            'gc_divisor' => 1000, // garbage collection on every 1000th session_start (gc_probability/gc_devisor)
            'gc_maxlifetime' => 14400 // 4 hours
        ];
        private $started = false;
        private $regenerate_after = 600;

        public function __construct()
        {
            if (\PHP_SESSION_NONE === session_status()) {
                foreach ($this->options as $k => $v) {
                    ini_set("session.$k", $v);
                }
            }
        }

        /**
         * Starts a session
         *
         * @return boolean
         * @throws \Exception
         */
        public function start(): bool
        {
            if ($this->started === true) {
                return true;
            }

            if (\PHP_SESSION_ACTIVE === session_status()) {
                $this->started = true;
                return true;
            }

            if (!session_start()) {
                throw new \Exception('Session could not be started');
            }
            $this->started = true;

            if (!isset($_SESSION['regenerated_time'])) {
                $this->set('regenerated_time', time());
            } else if (($_SESSION['regenerated_time'] + $this->regenerate_after) < time()) {
                $this->regenerate(false);
            }

            return true;
        }

        /**
         * Returns SESSION parameter
         * If no $key is passed returns entire $_SESSION array
         *
         * @param string $key Parameter's name
         * @return mixed
         * @throws \Exception
         */
        public function get(string $key = null)
        {
            if (!$this->started) {
                $this->start();
            }

            if ($key) {
                return $_SESSION[$key] ?? null;
            }

            return $_SESSION ?? [];
        }

        /**
         * Sets a parameter by name
         *
         * @param string $key Parameter's name
         * @param mixed $value
         * @throws \Exception
         */
        public function set(string $key, $value)
        {
            if (!$this->started) {
                $this->start();
            }
            $_SESSION[$key] = $value;
        }

        /**
         * Regenerates session ID and sets the time it was regenerated
         *
         * @param bool $destroy Whether to keep or destroy session data
         * @return boolean
         */
        public function regenerate(bool $destroy = false): bool
        {
            if (\PHP_SESSION_ACTIVE !== session_status()) {
                return false;
            }

            $regenerated = session_regenerate_id($destroy);

            if ($regenerated) {
                $_SESSION['regenerated_time'] = time();
            }

            return $regenerated;
        }

        /**
         * Closes a session
         */
        public function close()
        {
            session_write_close();
        }

        /**
         * Destroys a session
         */
        public function destroy()
        {
            if (!$this->started) {
                $this->start();
            }
            $_SESSION = [];
            $this->close();
            $this->started = false;
        }

    }