<?php

    namespace Asn\App\Model\AgencyPilot;

    use Asn\Core\Database;
    use Asn\App\Model\Location;

    class PropertyFinder
    {

        protected $db;
        protected $district;

        public function __construct(Database $db, Location $location)
        {
            $this->db = $db;
            $this->location = $location;
        }

        /**
         * Build query dynamically
         *
         * @param array $filters
         * @param int $start
         * @param int $perPage
         *
         * @return array
         * @throws \Throwable
         */
        public function buildQuery(array $filters, int $start, int $perPage): array
        {

            $sqlSelect = 'SELECT p._key, p.location_link, p.unit_name, p.town, p.post_code, p.size_dimensions, p.min_size, p.max_size, p.latitude, p.longitude, d.l2_name AS district';
            $sqlFrom = ' FROM properties AS p INNER JOIN districts_l2 AS d ON p.location_link = d.l2_no ';
            $where = '';
            $sqlHaving = '';
            $sqlParams = [];

            // this is `District`
            if (isset($filters['locationLink'])) {
                $where .= ' p.location_link = :location_link AND ';
                $sqlParams[] = [":location_link", $filters['locationLink']];
            } elseif (isset($filters['area'])) {
                $districtArr = $this->location->getDistricts((int) $filters['area']);

                $districtIn = "";

                foreach ($districtArr as $value) {
                    $districtIn .= $value['l2_no'] . ",";
                }

                $districtIn = rtrim($districtIn, ',');
                if ($districtIn !== '') {
                    $where .= " p.location_link IN ($districtIn) AND ";
                }
            }

            if (!empty($filters['propertyType'])) {
                $where .= ' p.unit_name = :unit_name AND ';
                $sqlParams[] = [":unit_name", $filters['propertyType']];
            }

            if (isset($filters['district'])) {
                $where .= ' p.district = :district AND ';
                $sqlParams[] = [":district", $filters['district']];
            }

            if (isset($filters['postCodeAddress'])) {
                // Check whether the full postcode is supplied or only the district part
                $postCode = $filters['postCodeAddress'];

                if (mb_strlen($postCode, 'UTF-8') <= 4) {
                    $where .= ' p.post_code_district = :post_code_district AND ';
                    $sqlParams[] = [":post_code_district", $postCode];
                } else {
                    $where .= ' p.post_code = :post_code AND ';
                    $sqlParams[] = [":post_code", $postCode];
                }
            }

            if (isset($filters['freehold']) || isset($filters['leasehold']) || isset($filters['long-leasehold'])) {
                $whereOr = '';

                if (isset($filters['freehold'])) {
                    $whereOr .= 'freehold = :freehold OR ';
                    $sqlParams[] = [":freehold", $filters['freehold']];
                }

                if (isset($filters['leasehold'])) {
                    $whereOr .= 'leasehold = :leasehold OR ';
                    $sqlParams[] = [":leasehold", $filters['leasehold']];
                }

                if (isset($filters['long-leasehold'])) {
                    $whereOr .= 'leasehold_term = :leasehold_term OR ';
                    $sqlParams[] = [":leasehold_term", $filters['long-leasehold']];
                }

                $whereOr = rtrim($whereOr, 'OR ');

                $where .= '(' . $whereOr . ') AND ';
            }

            if (isset($filters['size-dimension'])) {
                $where .= ' p.size_dimensions = :size_dimension AND ';
                $sqlParams[] = [":size_dimension", $filters['size-dimension']];

                $where .= ' p.min_size >= :min_size AND ';
                $sqlParams[] = [":min_size", $filters['size-min']];

                if (isset($filters['size-max'])) {

                    $where .= ' p.max_size <= :max_size AND ';
                    $sqlParams[] = [":max_size", $filters['size-max']];
                }
            }

            if (!empty($filters['postCode']) && isset($filters['distance'])) {
                try {
                    $latLong = $this->postCodeLatLong($filters['postCode']);
                } catch (\Throwable $th) {
                    throw $th;
                }

                $latitude = $latLong['latitude'];
                $longitude = $latLong['longitude'];
                $distance = $filters['distance'];

                $sqlSelect .= ', (
                    3959 * acos(
                        cos( radians(:latitude) ) * cos( radians( latitude ) ) * cos( radians( longitude )  - radians(:longitude) )
                        + sin( radians(:latitude2) ) * sin( radians( latitude ) )
                        )
                    )
                    AS distance ';

                $sqlHaving = ' HAVING distance < :distance';

                $sqlParams[] = [":latitude", $latitude];
                $sqlParams[] = [":longitude", $longitude];
                $sqlParams[] = [":latitude2", $latitude];
                $sqlParams[] = [":distance", $distance];
            }

            $where = rtrim($where, 'AND ');
            $where = rtrim($where, 'OR ');

            $sql = $sqlSelect . $sqlFrom;

            if ($where) {
                $sql .= ' WHERE' . $where;
            }

            $sql .= $sqlHaving;
            $sqlAll = $sql;

            $sql .= ' LIMIT ' . $start . ', ' . $perPage;

            return ['sql' => $sql, 'sqlAll' => $sqlAll, 'sqlParams' => $sqlParams];
        }

        /**
         * Return array with data from `properties` table, based on search query
         *
         * @param array $filters
         * @param int $start
         * @param int $perPage
         *
         * @return array
         * @throws \Throwable
         */
        public function find(array $filters, int $start, int $perPage): array
        {
            $brochuresArr = [];
            $photosArr = [];
            $photoWebArr = [];

            try {
                $queryData = $this->buildQuery($filters, $start, $perPage);

                $result = $this->db->exec($queryData['sql'], $queryData['sqlParams']);
                $arrKeys = $result->fetchAll();

                $resultAll = $this->db->exec($queryData['sqlAll'], $queryData['sqlParams']);
                $totalCount = $resultAll->rowCount();
            } catch (\Throwable $thr) {
                throw $thr;
            }

            $inClause = '';

            foreach ($arrKeys as $key => $value) {
                $_key = $value['_key'];
                $inClause .= "'$_key',";
            }

            $inClause = rtrim($inClause, ",");

            if ($inClause !== '') {
                try {
                    $result = $this->db->exec("SELECT _key, brochure_url FROM brochures WHERE _key IN ($inClause) ");

                    while ($row = $result->fetch()) {
                        $brochuresArr[$row['_key']][] = $row['brochure_url'];
                    }
                } catch (\Throwable $thr) {
                    throw $thr;
                }

                try {
                    $result = $this->db->exec("SELECT _key, all_photo_keys FROM photos WHERE _key IN ($inClause) ");

                    while ($row = $result->fetch()) {
                        $photosArr[$row['_key']] = explode(",", $row['all_photo_keys']);
                    }
                } catch (\Throwable $thr) {
                    throw $thr;
                }

                $result = $this->db->exec("SELECT _key, photo_key FROM photos WHERE _key IN ($inClause) ");
                while ($row = $result->fetch()) {
                    $photoWebArr[$row['_key']] = str_replace('_sm.jpg', '_web.jpg', $row['photo_key']);
                }
            }

            foreach ($arrKeys as $key => $arr) {
                $_key = $arr['_key'];

                if (isset($brochuresArr[$_key])) {
                    $arrKeys[$key]['brochures'] = $brochuresArr[$_key];
                }

                if (isset($photosArr[$_key])) {
                    $arrKeys[$key]['photos'] = $photosArr[$_key];
                }

                if (isset($photosArr[$_key])) {
                    $arrKeys[$key]['photo_web'] = $photoWebArr[$_key];
                }
            }

            return ['arrKeys' => $arrKeys, 'totalCount' => $totalCount];
        }

        /**
         * @param string $postCode
         *
         * @return array
         * @throws \Exception
         */
        public function postCodeLatLong(string $postCode): array
        {
            if (mb_strlen($postCode, 'UTF-8') <= 4) {
                $result = $this->db->exec("SELECT postcode, latitude, longitude
                    FROM open_postcode_geo
                    WHERE postcode_district = :postcode_district
                    LIMIT 1", [
                    [":postcode_district", $postCode]
                ]);
            } else {
                $result = $this->db->exec("SELECT postcode, latitude, longitude
                    FROM open_postcode_geo
                    WHERE postcode = :postcode
                    LIMIT 1", [
                    [":postcode", $postCode]
                ]);
            }
            $row = $result->fetch();
            if (!$row) {
                throw new \Exception('Postcode not found');
            }
            return ['latitude' => floatval($row['latitude']), 'longitude' => floatval($row['longitude'])];
        }

        /**
         * Return array with recently updated properties.
         * if $propertyType has been provided, return only one property type
         * for example: "Office".
         *
         * @param int $numResult
         * @param string $propertyType
         *
         * @return array
         * @throws \Throwable
         */
        public function getRecentlyUpdated(int $numResult, string $propertyType = ''): array
        {
            $brochuresArr = [];
            $photosArr = [];
            $photoWebArr = [];

            if (empty( $propertyType )) {
                $query = $this->db->exec( "SELECT
                                                    a._key, a.location_link, a.unit_name, a.town, a.post_code,
                                                    a.size_dimensions, a.min_size, a.max_size, a.latitude, a.longitude, a.updated,
                                                    b.l1_no, b.l1_name, b.l2_no, b.l2_name
                                                FROM
                                                    properties AS a
                                                INNER JOIN
                                                    districts_l2 AS b
                                                ON
                                                    a.location_link = b.l2_no
                                                ORDER BY
                                                    updated DESC
                                                LIMIT :limit_number",
                    [
                        [":limit_number", $numResult],
                    ]
                );
                $rows = $query->fetchAll();

            } else {
                $query = $this->db->exec( "SELECT
                                                    a._key, a.location_link, a.unit_name, a.town, a.post_code,
                                                    a.size_dimensions, a.min_size, a.max_size, a.latitude, a.longitude, a.updated,
                                                    b.l1_no, b.l1_name, b.l2_no, b.l2_name
                                                FROM
                                                    properties AS a
                                                INNER JOIN
                                                    districts_l2 AS b
                                                ON
                                                    a.location_link = b.l2_no
                                                WHERE
                                                    unit_name = :unit_name
                                                ORDER BY
                                                    updated DESC
                                                LIMIT :limit_number",
                    [
                        [":limit_number", $numResult],
                        [":unit_name", $propertyType]
                    ] );
                $rows = $query->fetchAll();

            }

            $inClause = '';

            foreach ($rows as $key => $value) {
                $_key = $value['_key'];
                $inClause .= "'$_key',";
            }

            $inClause = rtrim($inClause, ",");

            if ($inClause !== '') {
                try {
                    $result = $this->db->exec("SELECT _key, brochure_url FROM brochures WHERE _key IN ($inClause) ");

                    while ($row = $result->fetch()) {
                        $brochuresArr[$row['_key']][] = $row['brochure_url'];
                    }
                } catch (\Throwable $thr) {
                    throw $thr;
                }

                try {
                    $result = $this->db->exec("SELECT _key, all_photo_keys FROM photos WHERE _key IN ($inClause) ");

                    while ($row = $result->fetch()) {
                        $photosArr[$row['_key']] = explode(",", $row['all_photo_keys']);
                    }
                } catch (\Throwable $thr) {
                    throw $thr;
                }

                $result = $this->db->exec("SELECT _key, photo_key FROM photos WHERE _key IN ($inClause) ");
                while ($row = $result->fetch()) {
                    $photoWebArr[$row['_key']] = str_replace('_sm.jpg', '_web.jpg', $row['photo_key']);
                }
            }

            foreach ($rows as $key => $arr) {
                $_key = $arr['_key'];

                if (isset($brochuresArr[$_key])) {
                    $rows[$key]['brochures'] = $brochuresArr[$_key];
                }

                if (isset($photosArr[$_key])) {
                    $rows[$key]['photos'] = $photosArr[$_key];
                }

                if (isset($photosArr[$_key])) {
                    $rows[$key]['photo_web'] = $photoWebArr[$_key];
                }
            }

            return $rows;
        }

    }
