<?php

    namespace Asn\Core;

    use Asn\Core\Database;
    use Asn\Core\Encryption;
    use Asn\Core\Http\Session;
    use Asn\Core\Http\Request;

    class Auth
    {

        private $enc;
        protected $db;
        protected $session;
        protected $request;
        protected $user_role;

        /**
         *
         * @param Database $db
         * @param Encryption $enc
         * @param Session $session
         * @param Request $request
         */
        public function __construct(Database $db, Encryption $enc, Session $session, Request $request)
        {
            $this->db = $db;
            $this->enc = $enc;
            $this->session = $session;
            $this->request = $request;
        }

        /**
         *
         * @param string $firstName
         * @param string $lastName
         * @param string $username
         * @param string $pwd
         * @param string $role
         * @param int $status
         * @return array
         */
        public function createUser(string $firstName, string $lastName, string $username, string $pwd, string $role = 'user', int $status = 1): array
        {
            $firstName = trim($firstName);
            $lastName = trim($lastName);
            $username = trim($username);
            $role = trim($role);
            $pwd = password_hash(trim($pwd), PASSWORD_DEFAULT);

            try {
                $this->db->exec("INSERT INTO users (first_name, last_name, username, pwd, role, status, created)
                    VALUES (:first_name, :last_name, :username, :pwd, :role, :status, UNIX_TIMESTAMP())", [
                    [':first_name', $firstName],
                    [':last_name', $lastName],
                    [':username', $username],
                    [':pwd', $pwd],
                    [':role', $role],
                    [':status', $status]
                ]);
                $userId = (int) $this->db->lastInsertId();
            } catch (\Throwable $th) {
                if ((int) $th->getCode() === 23000) {
                    return ['success' => false, 'msg' => "User $username already exists"];
                } else {
                    return ['success' => false, 'msg' => 'An error occurred'];
                }
            }

            return ['success' => true, 'userId' => $userId, 'msg' => "User $username has been created"];
        }

        /**
         *
         * @param string $username
         * @param string $pwd
         * @param bool $remember
         * @return bool
         * @throws \Exception
         */
        public function logIn(string $username, string $pwd, bool $remember = false): bool
        {
            $username = trim($username);
            $pwd = trim($pwd);

            $res = $this->db->exec("SELECT user_id, pwd, role, status
					FROM users
					WHERE username = :username
					LIMIT 1", [
                [':username', $username]
            ]);

            if ($res->rowCount() === 0) {
                return false;
            }

            $row = $res->fetch();
            if ($row['status'] < 1 || $row['role'] !== $this->user_role || !password_verify($pwd, $row['pwd'])) {
                return false;
            }

            $userId = (int) $row['user_id'];
            $storageName = $this->storageName();

            if ($remember) {
                $expires = $expires_cookie = time() + 5 * 365 * 24 * 3600;
            } else {
                $expires_cookie = 0;
                $expires = time() + 24 * 3600;
            }

            $token = $this->generateToken();
            $validator = hash('SHA256', $token);

            try {
                $id = $userId . '_' . microtime();
                $this->db->exec("INSERT INTO users_auth(id, user_id, validator, expires)
                    VALUES (UNHEX(SHA1(:id)), :user_id, :validator, :expires)", [
                    [':id', $id],
                    [':user_id', $userId],
                    [':validator', $validator],
                    [':expires', $expires]
                ]);
            } catch (\Throwable $th) {
                return false;
            }

            $this->updateLastLogin($userId);
            $this->cleanAuthTable($userId);

            $this->session->set($storageName, $userId);

            $cookie = base64_encode($this->enc->encrypt($userId . ':' . $token));
            $domain = $this->cookieDomain();
            return setcookie($storageName, $cookie, $expires_cookie, '/', $domain, false, true);
        }

        /**
         * Deletes redundant validator tokens
         *
         * @param int $userId
         * @throws \Exception
         */
        public function cleanAuthTable(int $userId = null)
        {
            if ($userId) {
                $this->db->exec("DELETE FROM users_auth
                    WHERE user_id = :user_id AND expires < UNIX_TIMESTAMP()", [
                    [':user_id', $userId]
                ]);
            } else {
                $this->db->exec("DELETE FROM users_auth
                    WHERE expires < UNIX_TIMESTAMP()");
            }
        }

        /**
         * Checks whether the user is logged
         * On success, SESSION is always closed for performance reasons
         *
         * @return boolean|int
         */
        protected function isAuth()
        {
            $storageName = $this->storageName();

            $id = $this->session->get($storageName);
            if (is_numeric($id)) {
                $this->session->close();
                return $id;
            }

            $cookieAuthRow = $this->cookieAuthRow($storageName);
            if (!$cookieAuthRow) {
                return false;
            }
            $userId = $cookieAuthRow[1];

            $this->session->set($storageName, $userId);
            $this->session->close();

            $this->updateLastLogin($userId);
            return $userId;
        }

        /**
         *
         * @param int $userId
         * @param string $pwd
         * @param bool $login
         * @return bool
         */
        protected function resetPasword(int $userId, string $pwd, bool $login = false): bool
        {
            if ($pwd === '') {
                return false;
            }
            $pwd_hash = password_hash(trim($pwd), PASSWORD_DEFAULT);
            try {
                $this->db->exec('UPDATE users
                    SET pwd = :pwd
                    WHERE user_id = :user_id
                    LIMIT 1', [
                    [':pwd', $pwd_hash],
                    [':user_id', $userId]
                ]);

                if ($login) {
                    $res = $this->db->exec('SELECT username
                        FROM users
                        WHERE user_id=:user_id
                        LIMIT 1', [
                        [':user_id', $userId]
                    ]);
                    $row = $res->fetch();
                    return $this->logIn($row['username'], $pwd);
                }

                return true;
            } catch (\Throwable $th) {
                return false;
            }
        }

        /**
         *
         * @param int $userId
         * @param string $pwd
         * @return bool
         * @throws \Exception
         */
        public function verifyPassword(int $userId, string $pwd): bool
        {
            $res = $this->db->exec('SELECT pwd
                FROM users
                WHERE user_id = :user_id
                LIMIT 1', [
                [':user_id', $userId]
            ]);
            $row = $res->fetch();
            return password_verify($pwd, $row['pwd']);
        }

        /**
         *
         * @return bool
         * @throws \Exception
         */
        public function logOut(): bool
        {
            $storageName = $this->storageName();

            $cookieAuthRow = $this->cookieAuthRow($storageName);

            if ($cookieAuthRow) {
                $id = $cookieAuthRow[0];
                $userId = $cookieAuthRow[1];
                $this->db->exec("DELETE FROM users_auth
                    WHERE user_id = :user_id AND HEX(id) = :id
                    LIMIT 1", [
                    [':user_id', $userId],
                    [':id', $id]
                ]);
            }

            $this->session->start();
            unset($_SESSION[$storageName]);
            $this->session->close();

            $domain = $this->cookieDomain();
            return setcookie($storageName, '', time() - 3600, '/', $domain, false, true);
        }

        /**
         *
         * @param int $userId
         * @throws \Exception
         */
        private function updateLastLogin(int $userId)
        {
            $this->db->exec("UPDATE users
                SET last_login = UNIX_TIMESTAMP()
		WHERE user_id = :user_id
		LIMIT 1", [
                [':user_id', $userId]
            ]);
        }

        /**
         *
         * @return string
         * @throws \Exception
         */
        private function generateToken(): string
        {
            return bin2hex(random_bytes(20));
        }

        /**
         * Returns the domain to be used for setcookie function
         *
         * @return string
         */
        private function cookieDomain(): string
        {
            $d = $this->request->getServerVar('SERVER_NAME');
            return (mb_stripos($d, 'www.', 0, 'UTF-8') === 0) ? mb_substr($d, 4, null, 'UTF-8') : $d;
        }

        /**
         * SESSION/COOKIE storage name
         *
         * @return string
         */
        private function storageName(): string
        {
            return ($this->user_role === 'admin') ? 'a' : 'b';
        }

        /**
         * Returns id and user_id from auth_user identified using COOKIE data
         *
         * @param string $storageName
         * @return array|null
         * @throws \Exception
         */
        private function cookieAuthRow(string $storageName)
        {
            $cookie64 = $this->request->getCookie($storageName);
            if (!$cookie64 || !($cookie = base64_decode($cookie64, true))) {
                return null;
            }

            $decrypted = $this->enc->decrypt($cookie);
            if ($decrypted === false) {
                return null;
            }

            $cookieData = explode(':', $decrypted);
            if (count($cookieData) !== 2 || !is_numeric($cookieData[0])) {
                return null;
            }

            $cookieId = (int) $cookieData[0];

            $res = $this->db->exec("SELECT HEX(id) AS id, user_id, validator
		FROM users_auth
		WHERE user_id = :user_id > 0 AND expires > UNIX_TIMESTAMP()", [
                [':user_id', $cookieId]
            ]);

            if ($res->rowCount() === 0) {
                return null;
            }

            $rows = $res->fetchAll();
            foreach ($rows as $row) {
                $userId = (int) $row['user_id'];
                if ($userId === $cookieId && hash_equals($row['validator'], hash('SHA256', $cookieData[1]))) {
                    return [$row['id'], $userId];
                }
            }

            return null;
        }

    }
