// Global variables
var xhrPool = []; // AJAX requests pool

/**
 * Makes an ajax call
 *
 * @param {object} settings
 * @param {string} settings.URL
 * @param {?boolean} settings.addToPool
 * @param {?boolean} settings.abortPending
 * @param {string} settings.reqMethod [POST|GET]
 * @param {string} settings.reqType [formdata|json]
 * @param {object} settings.reqParams [null]
 * @param {string} settings.respType [json|text]
 * @param {object} settings.callback 
 */

function ajax(settings) {

    if (settings.abortPending === true) {
        for (let i = 0, len = xhrPool.length; i < len; i++) {
            if (xhrPool[i]) {
                xhrPool[i].abort();
            }
        }
        xhrPool.length = 0;
    }

    return new Promise((resolve, reject) => {
        let xhr = new XMLHttpRequest(),
            strURL = settings.URL,
            reqMethod = settings.reqMethod || 'POST',
            reqType = settings.reqType || 'formdata',
            respType = settings.respType || 'json',
            reqParams = typeof settings.reqParams === 'object' ? settings.reqParams : null;

        if (reqParams) {
            if (reqMethod === 'GET') {
                let qs = '',
                    rqKeys = Object.keys(reqParams);
                for (let i = 0, iLen = rqKeys.length; i < iLen; i++) {
                    qs += encodeURIComponent(rqKeys[i]) + "=" + encodeURIComponent(reqParams[rqKeys[i]]) + "&";
                }

                if (qs) {
                    strURL += "?" + qs.slice(0, -1);
                }

            } else if (reqType === 'json') {
                reqParams = JSON.stringify(reqParams);
            }
        }

        xhr.onreadystatechange = function () {
            if (xhr.readyState === 1 && settings.addToPool) {
                xhrPool.push(xhr);
            }

            if (xhr.readyState === 4) {
                if (settings.addToPool) {
                    let hxrIndex = xhrPool.indexOf(xhr);
                    if (hxrIndex > -1) {
                        xhrPool.splice(hxrIndex, 1);
                    }
                }

                if (xhr.status === 200) {
                    let resp = xhr.response;
                    if (respType === 'json') {
                        // Needed for IE only as it return sring instead of JSON
                        resp = xhr.responseType === 'json' ? xhr.response : JSON.parse(xhr.responseText);
                        if (typeof resp !== 'object' || resp === null) {
                            reject(resp);
                        }
                    }
                    resolve(resp);
                } else {
                    reject(xhr.status); 
                }
            }
        };

        xhr.open(reqMethod, strURL, true);

        if (reqType === 'json') {
            xhr.setRequestHeader("Content-Type", "application/json");
        }
        xhr.responseType = respType;
        xhr.send(reqParams);
    });
}

/**
 * Check if elem is an HTML element
 * @param {object} elem
 * @return {Boolean}
 */
function isHtmlElement(elem) {
    return elem && typeof elem === 'object' && elem.nodeType === 1;
}

/**
 * Returns HTML element from ID or node element.
 * Undefined if the element does not exist
 *
 * @param {string|object} elemRef
 * @return {object|undefined}
 */
function getHtmlElement(elemRef) {
    let elem;
    if (typeof elemRef === 'string') {
        elem = document.getElementById(elemRef);
    } else if (elemRef && typeof elemRef === 'object' && elemRef.nodeType === 1) {
        elem = elemRef;
    }
    return elem;
}

/**
 * Check if a string is numeric
 *
 * @param {string} val
 * @return {Boolean}
 */
function isNumeric(val) {
    return !isNaN(parseFloat(val)) && isFinite(val);
}

/**
 * Check if a string is "empty"
 *
 * @param {string} val
 * @return {Boolean}
 */
function isEmpty(val) {
    return !val || (typeof val === 'string' && val.trim() === '');
}

/**
 * Returns the first parent element of the specified class or node name
 *
 * @param {object} elem
 * @param {string} name class name or node(tag) name
 * @return {object|undefined}
 */
function getParentElement(elem, name) {
    if (!isHtmlElement(elem) || isEmpty(name)) {
        return;
    }

    if (name.slice(0, 1) === '.') { // Class name
        name = name.slice(1);
        while (elem.parentNode) {
            elem = elem.parentNode;
            try {
                if (elem.classList.contains(name)) {
                    return elem;
                }
            } catch (err) {
                return;
            }
        }
    } else { // Tag name
        name = name.toUpperCase();
        while (elem.parentNode) {
            elem = elem.parentNode;
            if (elem.nodeName === name) {
                return elem;
            }
        }
    }
}

/**
 *
 * Append options to select element
 *
 * @param {object|string} elemRef HTML element or id
 * @param {array} opts
 */
function selectAppendOptions(elemRef, opts) {
    let elem = getHtmlElement(elemRef);

    if (elem && Array.isArray(opts)) {
        const doc = document,
            frag = doc.createDocumentFragment();

        let opt,
            optAttr;

        for (let i = 0, iLen = opts.length; i < iLen; i++) {
            opt = doc.createElement('option');
            if (opts[i].attr) {
                optAttr = opts[i].attr;
                opt.setAttribute(optAttr, optAttr);
            }
            opt.value = opts[i].val;
            opt.text = opts[i].txt;
            frag.appendChild(opt);
        }
        elem.appendChild(frag);
    }
}

/**
 * Remove options from select element, but keep first k options
 *
 * @param {object} elemRef HTML element or id
 * @param {int} k number of options to keep
 */
function selectRemoveOptions(elemRef, k) {
    let elem = getHtmlElement(elemRef);
    if (elem) {
        let keep = (isNumeric(k) && k >= 0) ? k : 0,
            i = elem.options.length;
        while (i-- > keep) {
            elem.remove(i);
        }
    }
}

/**
 *
 * @param {type} propertyType
 * @param {type} filterId - the form id
 * @param {type} page - page number
 * @param {type} perPage - records per page
 * @returns {Promise}
 */
function propertyFinder(propertyType, filterId, page, perPage) {

    const doc = document;
    let   form = doc.getElementById(filterId),
        searchResContainer = doc.getElementById('search-results'),
        oFD = form ? new FormData(form) : new FormData();

    oFD.append("propertyType", propertyType);
    oFD.append("filterId", filterId);
    oFD.append("page", page);
    oFD.append("perPage", perPage);

    return ajax({
        "URL": "/ajax/find-properties.php",
        "reqParams": oFD
    }).then((resp) => {

        if (resp.success === true) {

            searchResContainer.innerHTML = resp.result;
        } else {
            searchResContainer.textContent = '';
        }

        const msgE = doc.getElementById('msg');
        if (msgE && resp.msg) {
            msgE.textContent = resp.msg;
        }
    });
}

function searchOffice() {
    const doc = document,
        areasE = doc.getElementById('area'),
        districtsE = doc.getElementById('district'),
        msgE = doc.getElementById('msg');


    // get district by selected area
    areasE.addEventListener("change", (e) => {
        selectRemoveOptions(districtsE, 2);
        ajax({
            "URL": "/ajax/location/get-districts.php",
            "reqMethod": "POST",
            "reqType": "json",
            "reqParams": {'areaId': e.currentTarget.value}
        }).then((resp) =>
        {
            if (resp.success === true)
            {
                selectAppendOptions(districtsE, resp.districtsOptions);
            }
        });
    });

    // attachSearchByEvents
    let buttonsE = doc.getElementById('search-fiters-wrapper').children;
    let formsE = doc.getElementsByClassName('filter-form'),
        searchResultsE = doc.getElementById('search-results');

    for (let i = 0, len = buttonsE.length; i < len; i++)
    {
        searchResultsE.textContent = '';

        buttonsE[i].addEventListener('click', (e) =>
        {
            searchResultsE.textContent = '';
            msgE.textContent = '';

            for (let j = 0, len1 = formsE.length; j < len1; j++)
            {
                if (e.currentTarget.getAttribute('data-id') === formsE[j].getAttribute('id'))
                {
                    formsE[j].classList.add('is-active');
                } else
                {
                    formsE[j].classList.remove('is-active');
                }
            }
        }, false);
    }

    // attachSearchButtonEvents
    let searchButtonsE = doc.getElementsByClassName('property-search-button'),
        len = searchButtonsE.length;

    while (len--)
    {
        searchButtonsE[len].addEventListener('click', (e) =>
        {
            let formE = getParentElement(e.currentTarget, 'form');

            if (formE.id === 'office-by-distance-form') {
                const distanceE = formE['distance'],
                    postcodeE = formE['postcode-by-distance'];
                if (!distanceE || !postcodeE) {
                    console.log('missing form elements');
                    return;
                }

                if (isEmpty(distanceE.value) || isEmpty(postcodeE.value)) {
                    msgE.textContent = 'Required fields are left blank';
                    return;
                }
                msgE.textContent = '';
            }

            // ajax call function
            propertyFinder('Office', formE.id);
        });
    }

    //attach events to select elements
    let selectElementsE = doc.getElementsByClassName('select-area'),
        lenS = selectElementsE.length;

    while (lenS--)
    {
        selectElementsE[lenS].addEventListener('change', (e) =>
        {
            let selectedId = e.currentTarget.getAttribute('id');
            let lenS1 = selectElementsE.length;
            while (lenS1--)
            {
                if (selectElementsE[lenS1].id !== selectedId)
                {
                    selectElementsE[lenS1].selectedIndex = "0";
                }
            }
        }, false);
    }
}

function searchRetail()
{
    const doc = document,
        areasE = doc.getElementById('area'),
        districtsE = doc.getElementById('district'),
        msgE = doc.getElementById('msg');

    // get district by selected area
    areasE.addEventListener("change", (e) => {
        selectRemoveOptions(districtsE, 2);
        ajax({
            "URL": "/ajax/location/get-districts.php",
            "reqMethod": "POST",
            "reqType": "json",
            "reqParams": {'areaId': e.currentTarget.value}
        }).then((resp) =>
        {
            if (resp.success === true)
            {
                selectAppendOptions(districtsE, resp.districtsOptions);
            }
        });
    });

    // attachSearchByEvents
    let buttonsE = doc.getElementById('search-fiters-wrapper').children,
        formsE = doc.getElementsByClassName('filter-form'),
        searchResultsE = doc.getElementById('search-results');

    for (let i = 0, len = buttonsE.length; i < len; i++)
    {
        buttonsE[i].addEventListener('click', (e) =>
        {
            searchResultsE.textContent = '';
            msgE.textContent = '';

            for (let j = 0, len1 = formsE.length; j < len1; j++)
            {
                if (e.currentTarget.getAttribute('data-id') === formsE[j].getAttribute('id'))
                {
                    formsE[j].classList.add('is-active');
                } else
                {
                    formsE[j].classList.remove('is-active');
                }
            }
        }, false);
    }

    // attachSearchButtonEvents
    let searchButtons = doc.getElementsByClassName('property-search-button'),
        len = searchButtons.length;

    while (len--)
    {
        searchButtons[len].addEventListener('click', (e) =>
        {
            let formE = getParentElement(e.currentTarget, 'form');

            if (formE.id === 'retail-by-distance-form') {
                const distanceE = formE['distance'],
                    postcodeE = formE['postcode-by-distance'];
                if (!distanceE || !postcodeE) {
                    console.log('missing form elements');
                    return;
                }

                if (isEmpty(distanceE.value) || isEmpty(postcodeE.value)) {
                    msgE.textContent = 'Required fields are left blank';
                    return;
                }
                msgE.textContent = '';
            }

            // ajax call function
            propertyFinder('Retail', formE.id);
        });
    }

    //attach events to select elements
    let selectElementsE = doc.getElementsByClassName('select-area'),
        lenS = selectElementsE.length;

    while (lenS--)
    {
        selectElementsE[lenS].addEventListener('change', (e) =>
        {
            let selectedId = e.currentTarget.getAttribute('id');
            let lenS1 = selectElementsE.length;
            while (lenS1--)
            {
                if (selectElementsE[lenS1].id !== selectedId)
                {
                    selectElementsE[lenS1].selectedIndex = "0";
                }
            }
        }, false);
    }
}

function searchIndustrial()
{
    const doc = document,
        areasE = doc.getElementById('area'),
        districtsE = doc.getElementById('district'),
        msgE = doc.getElementById('msg');

    // get district by selected area
    areasE.addEventListener("change", (e) => {
        selectRemoveOptions(districtsE, 2);
        ajax({
            "URL": "/ajax/location/get-districts.php",
            "reqMethod": "POST",
            "reqType": "json",
            "reqParams": {'areaId': e.currentTarget.value}
        }).then((resp) =>
        {
            if (resp.success === true)
            {
                selectAppendOptions(districtsE, resp.districtsOptions);
            }
        });
    });

    // attachSearchByEvents
    let buttonsE = doc.getElementById('search-fiters-wrapper').children,
        formsE = doc.getElementsByClassName('filter-form'),
        searchResultsE = doc.getElementById('search-results');

    for (let i = 0, len = buttonsE.length; i < len; i++)
    {
        buttonsE[i].addEventListener('click', (e) =>
        {
            searchResultsE.textContent = '';
            msgE.textContent = '';

            for (let j = 0, len1 = formsE.length; j < len1; j++)
            {
                if (e.currentTarget.getAttribute('data-id') === formsE[j].getAttribute('id'))
                {
                    formsE[j].classList.add('is-active');
                } else
                {
                    formsE[j].classList.remove('is-active');
                }
            }
        }, false);
    }

    // attachSearchButtonEvents:
    let searchButtonsE = doc.getElementsByClassName('property-search-button'),
        len = searchButtonsE.length;

    while (len--) {
        searchButtonsE[len].addEventListener('click', (e) =>
        {
            let formE = getParentElement(e.currentTarget, 'form');

            if (formE.id === 'industrial-by-distance-form') {
                const distanceE = formE['distance'],
                    postcodeE = formE['postcode-by-distance'];
                if (!distanceE || !postcodeE) {
                    console.log('missing form elements');
                    return;
                }

                if (isEmpty(distanceE.value) || isEmpty(postcodeE.value)) {
                    msgE.textContent = 'Required fields are left blank';
                    return;
                }
                msgE.textContent = '';
            }

            // ajax call function
            propertyFinder('Industrial', formE.id);
        });
    }

    // attach events to select elements
    let selectElementsE = doc.getElementsByClassName('select-area'),
        lenS = selectElementsE.length;

    while (lenS--) {
        selectElementsE[lenS].addEventListener('change', (e) =>
        {
            let selectedId = e.currentTarget.getAttribute('id');
            let lenS1 = selectElementsE.length;

            while (lenS1--)
            {
                if (selectElementsE[lenS1].id !== selectedId)
                {
                    selectElementsE[lenS1].selectedIndex = "0";
                }
            }
        }, false);
    }

}

function SearchByPropertyType() {
    const doc = document;
    let searchByPropTypeFrmE = doc.getElementById('search-by-property-type'),
        searchByPropTypeBtnE = doc.getElementById('search-by-property-type-btn'),
        searchResContainer = doc.getElementById('search-results'),
        oFD = searchByPropTypeFrmE ? new FormData(searchByPropTypeFrmE) : new FormData();

    searchByPropTypeBtnE.addEventListener("click", () => {
        ajax({
            "URL": "ajax/find-properties.php",
            "reqParams": oFD,
            "reqType": 'formdata'
        }).then((resp) => {

            if (resp.success === true) {
                searchResContainer.innerHTML = resp.result;
            } else {
                searchResContainer.textContent = '';
            }
            const msgE = doc.getElementById('msg');
            if (msgE && resp.msg) {
                msgE.textContent = resp.msg;
            }
        });
    }, false);
}

document.addEventListener('DOMContentLoaded', () => {
    const doc = document;

    const initFuncs = {
        "common": async () => {
        },
        "searchOfficeInit": async () => {
            searchOffice();
        },
        "searchRetailInit": async () => {
            searchRetail();
        },
        "searchIndustrialInit": async () => {
            searchIndustrial();
        },
        "searchInvestmentInit": async () => {
            propertyFinder('investment', 'investment-form');
        },
        "searchLeisureInit": async () => {
            propertyFinder('leisure', 'leisure-form');
        },
        "searchResidentialInit": async () => {
            propertyFinder('residential', 'residential-form');
        },
        "searchDevelopmentInit": async () => {
            propertyFinder('development', 'development-form');
        },
        "searchByPropertyTypeInit": async () => {
            SearchByPropertyType();
        }
    };

    initFuncs.common.call();
    const initFn = doc.body.getAttribute('data-init');
    if (typeof initFuncs[initFn] === 'function') {
        initFuncs[initFn].call();
    }
}, false);
