<?php

    namespace Asn\App\Library;

    use Asn\Core\Http\Request;

    class Pagination
    {

        /**
         * Create pagination functionality in ajax file
         *
         * @param int $cnt 		- total count of pages
         * @param string $fn 	- js function name, that makes ajax call
         * @param array $fnP	- js function params (optional)
         * @param int $page		- page start from 1 (by default)
         * @param int $perPage	- number of pages per page
         * @param int $numLinks - how many link can be displayed, if is set to 4: (1234)
         *
         * @return string
         */
        public function paginationAjax(int $cnt, string $fn, array $fnP = [], int $page = 1, int $perPage = 25, int $numLinks = 10): string
        {
            $startPage = 1;
            $links = '';
            $numPages = ceil($cnt / $perPage);

            if (!$fn) {
                return '';
            }
            if ($numLinks < 3) {
                $numLinks = 3;
            }
            if ($cnt > $perPage) {
                if ($page == 0) {
                    $page = 1;
                }
                $startPage = $page;
                $numPages = ceil($cnt / $perPage);
                $links = '';
                $fnParams = implode(',', $fnP);
                if ($fnParams) {
                    $fnParams .= ',';
                }
                if ($numPages <= $numLinks) {
                    $startPage = 1;
                } else {
                    $startPage = $page - floor($numLinks / 2);
                    if ($startPage <= 0) {
                        $startPage = 1;
                    } elseif ($numPages - $startPage < $numLinks) {
                        $startPage = $numPages - $numLinks + 1;
                    }
                }

                for ($pg = $startPage; $pg < ($startPage + $numLinks) && $pg <= $numPages; $pg++) {
                    $links = ($page != $pg) ? "$links<span onclick=\"$fn($fnParams $pg,$perPage,$numLinks);\">$pg</span>" : "$links<span class=\"curr\">$pg</span>";
                }
                if ($page > 1 && $numPages > 1) {
                    $prev = $page - 1;
                    $links = "<span id=\"prv\" onclick=\"$fn($fnParams $prev,$perPage,$numLinks);\">Previous</span>" . $links;
                }
                if ($page < $numPages) {
                    $next = $page + 1;
                    $links .= "<span id=\"nxt\" onclick=\"$fn($fnParams $next,$perPage,$numLinks);\">Next</span>";
                }
            }
            return $links;
        }

        /**
         * Returns html with pagination links
         *
         * @param int $cnt
         * @param int $page
         * @param int $perPage
         * @param int $numLinks
         *
         * @return string
         */
        public function pagination(int $cnt, int $page = 1, int $perPage = 30, int $numLinks = 10): string
        {
            $request = new Request();
            $links = '';
            if ($numLinks < 3) {
                $numLinks = 3;
            }
            if ($cnt > $perPage) {
                $url = $request->getServerVar('REQUEST_URI');
                if ($q = mb_strrpos($url, '/page', 0, 'UTF-8')) {
                    $url = mb_substr($url, 0, $q, 'UTF-8');
                }
                if ($page === 0) {
                    $page = 1;
                }
                $startPage = $page;
                $numPages = ceil($cnt / $perPage);
                if ($numPages <= $numLinks) {
                    $startPage = 1;
                } else {
                    $startPage = $page - floor($numLinks / 2);
                    if ($startPage <= 0) {
                        $startPage = 1;
                    } elseif ($numPages - $startPage < $numLinks) {
                        $startPage = $numPages - $numLinks + 1;
                    }
                    for ($pg = $startPage; $pg < ($startPage + $numLinks) && $pg <= $numPages; $pg++) {
                        $param = ($pg != 1) ? "/page=$pg" : '';
                        $links = ($page != $pg) ? "$links<a href=\"$url$param\">$pg</a>&nbsp;" : "$links<span class=\"curr\">$pg</span>&nbsp;";
                    }
                    if ($page > 1 && $numPages > 1) {
                        $prev = $page - 1;
                        $prev = ($prev != 1) ? "/page=$prev" : '';
                        $links = "<a href=\"$url$prev\">Previous</a>&nbsp;&nbsp;&nbsp;" . $links;
                    }
                    if ($page < $numPages) {
                        $next = $page + 1;
                        $next = "/page=$next";
                        $links .= "&nbsp;&nbsp;&nbsp;<a href=\"$url$next\">Next</a>";
                    }
                }
            }
            return $links;
        }

    }
