<?php

    namespace Asn\Core;

    use Asn\App\Bootstrap;
    use Asn\Core\Security;

    class Renderer
    {

        protected $bootstrap;
        protected $security;
        protected $cssFiles = [];
        protected $jsFiles = [];
        protected $layout = 'layout';
        protected $layoutPath = '';
        protected $tplPath = '';
        protected $tplData = [];
        private $appBase = '';
        private $sectionBase = '';

        /**
         *
         * @param Bootstrap $bootsrap
         * @param Security $security
         * @throws \Exception
         */
        public function __construct(Bootstrap $bootsrap, Security $security)
        {
            $this->bootstrap = $bootsrap;
            $this->security = $security;

            $this->appBase = $bootsrap->getAppBase();
            $this->sectionBase = $bootsrap->getSectionBase();

            $sectionData = $this->bootstrap->getSectionData();
            foreach ($sectionData as $k => $v) {
                $this->addData($k, $v);
            }

            $sectionSetings = $bootsrap->getDefalutSettings();
            if (isset($sectionSetings['css'])) {
                foreach ($sectionSetings['css'] as $css) {
                    $this->addCss($css);
                }
            }

            if (isset($sectionSetings['js'])) {
                foreach ($sectionSetings['js'] as $js) {
                    $src = $js[0];
                    $execMode = (isset($js[1])) ? $js[1] : '';
                    $this->addJs($src, $execMode);
                }
            }

            if (isset($sectionSetings['layout'])) {
                $this->layout = $sectionSetings['layout'];
            }
        }

        /**
         * Sets the layout template
         *
         * @param string $layout The layout template
         * @return $this
         * @throws \Exception
         */
        public function setLayout(string $layout)
        {
            $layoutPath = __DIR__ . '/../templates' . $this->sectionBase . '/' . $layout . '.lt.php';
            if (!file_exists($layoutPath)) {
                throw new \Exception("Layout $layoutPath not found");
            }
            $this->layout = $layout;
            $this->layoutPath = $layoutPath;
            return $this;
        }

        /**
         * Sets the content template
         *
         * @param string $tpl The content template
         * @return $this
         * @throws \Exception
         */
        public function setTemplate(string $tpl)
        {
            $tplPath = __DIR__ . '/../templates' . $this->sectionBase . '/tpl/' . $tpl . '.tpl.php';
            if (!file_exists($tplPath)) {
                throw new \Exception("Template $tplPath not found");
            }
            $this->tplPath = $tplPath;
            return $this;
        }

        /**
         * Renders the page
         */
        public function render(string $tpl)
        {
            if (!$this->layoutPath) {
                $this->setLayout($this->layout);
            }

            $this->setTemplate($tpl);
            $tplPath = $this->tplPath;

            $tplData = $this->tplData;

            include $this->layoutPath;
        }

        /**
         * Adds data to content template
         *
         * @param string|int $key
         * @param int|float|string|array|bool|null $data
         * @param string $escape Escape mode: html, html_all, raw
         *
         * @return $this
         * @throws \Exception
         */
        public function addData($key, $data, $escape = 'html')
        {
            if (!is_scalar($data) && !is_array($data) && !is_null($data)) {
                throw new \Exception('Data type not supported');
            }

            $escapeMethod = '';
            if ($escape === 'html') {
                $escapeMethod = 'encodeSpecialChars';
            } else if ($escape === 'html_all') {
                $escapeMethod = 'encodeHtmlEntities';
            } else if ($escape !== 'raw') {
                throw new \Exception('Invalid escape flag');
            }

            if ($escapeMethod) {
                if (is_string($data)) {
                    $data = $this->security->$escapeMethod($data);
                } else if (is_array($data)) {
                    array_walk_recursive($data, function(&$v) use ($escapeMethod) {
                        if (is_string($v)) {
                            $v = $this->security->$escapeMethod($v);
                        }
                    });
                }
            }

            $this->tplData[$key] = $data;
            return $this;
        }

        /**
         * Returns content template data
         *
         * @return array
         */
        public function getData(string $key = null): array
        {
            if ($key) {
                return $this->tplData[$key] ?? null;
            }
            return $this->tplData;
        }

        /**
         * Adds a css file to be included in the page
         *
         * @param string $path Path to the css file
         *
         * @return $this
         */
        public function addCss(string $path)
        {
            $this->cssFiles[] = $path;
            return $this;
        }

        /**
         * Returns html for adding css files to the page's head
         *
         * @return string $cssHTML
         */
        public function renderCss(): string
        {
            $cssHTML = '';
            foreach ($this->cssFiles as $path) {
                $cssHTML .= '<link href="' . $this->appBase . '/assets/' . $path . '" rel="stylesheet" type="text/css">' . "\n";
            }
            return $cssHTML;
        }

        /**
         * Adds a JavaScript file to be included in the page
         *
         * @param string $src Script's source
         * @param string $execMode Loading/execution mode (async, defer)
         *
         * @return $this
         */
        public function addJs(string $src, string $execMode = '')
        {
            $this->jsFiles[] = [
                'src' => $src,
                'execMode' => $execMode
            ];
            return $this;
        }

        /**
         * Returns html for adding JavaScripts files to the page
         *
         * @return string $jsHTML
         */
        public function renderJs(): string
        {
            $jsHTML = '';
            foreach ($this->jsFiles as $arr) {
                $jsHTML .= '<script src="' . $this->appBase . '/assets/' . $arr['src'] . '"';
                if ($arr['execMode'] !== '') {
                    $jsHTML .= ' ' . $arr['execMode'];
                }
                $jsHTML .= "></script>\n";
            }

            return $jsHTML;
        }

    }