<?php

    namespace Asn\Core;

    use Asn\Core\Http\Session;

    class Security
    {

        private $hmacKey;
        private $session;
        private $scrfSessionIndex = 'csrf';
        private $xssPreg = [
            // &entity
            '!(&#0+[0-9]+)!' => '$1;',
            '/(&#*\w+)[\x00-\x20]+;/u' => '$1;>',
            '/(&#x*[0-9A-F]+);*/iu' => '$1;',
            //any attribute starting with "on" or xml name space
            '#(<[^>]+?[\x00-\x20"\'])(?:on|xmlns)[^>]*+>#iu' => '$1>',
            //javascript: and VB script: protocols
            '#([a-z]*)[\x00-\x20]*=[\x00-\x20]*([`\'"]*)[\x00-\x20]*j[\x00-\x20]*a[\x00-\x20]*v[\x00-\x20]*a[\x00-\x20]*s[\x00-\x20]*c[\x00-\x20]*r[\x00-\x20]*i[\x00-\x20]*p[\x00-\x20]*t[\x00-\x20]*:#iu' => '$1=$2nojavascript...',
            '#([a-z]*)[\x00-\x20]*=([\'"]*)[\x00-\x20]*v[\x00-\x20]*b[\x00-\x20]*s[\x00-\x20]*c[\x00-\x20]*r[\x00-\x20]*i[\x00-\x20]*p[\x00-\x20]*t[\x00-\x20]*:#iu' => '$1=$2novbscript...',
            '#([a-z]*)[\x00-\x20]*=([\'"]*)[\x00-\x20]*-moz-binding[\x00-\x20]*:#u' => '$1=$2nomozbinding...',
            // Only works in IE: <span style="width: expression(alert('Ping!'));"></span>
            '#(<[^>]+?)style[\x00-\x20]*=[\x00-\x20]*[`\'"]*.*?expression[\x00-\x20]*\([^>]*+>#i' => '$1>',
            '#(<[^>]+?)style[\x00-\x20]*=[\x00-\x20]*[`\'"]*.*?s[\x00-\x20]*c[\x00-\x20]*r[\x00-\x20]*i[\x00-\x20]*p[\x00-\x20]*t[\x00-\x20]*:*[^>]*+>#iu' => '$1>',
            // namespace elements
            '#</*\w+:\w[^>]*+>#i' => '',
            //unwanted tags
            '#</*(?:applet|b(?:ase|gsound|link)|embed|frame(?:set)?|i(?:frame|layer)|l(?:ayer|ink)|meta|object|s(?:cript|tyle)|title|xml)[^>]*+>#i' => '',
            // php and javascript commands
            '#(alert|prompt|confirm|cmd|passthru|eval|exec|expression|system|fopen|fsockopen|file|file_get_contents|readfile|unlink)(\s*)\((.*?)\)#si' => '\\1\\2&#40;\\3&#41;'
        ];

        public function __construct(Session $session)
        {
            $settings = @include __DIR__ . '/../config/security.config.php';
            if (!is_array($settings)) {
                exit();
            }

            $this->hmacKey = base64_decode($settings['hmacKey']);

            $this->session = $session;
        }

        /**
         *
         * @param bool $deleteToken
         * @return bool
         */
        public function csrfVerify(array $post, bool $deleteToken = false): bool
        {
            $csrfArr = $this->session->get($this->scrfSessionIndex);
            if (!is_array($csrfArr)) {
                return false;
            }

            foreach ($post as $k => $v) {
                if (isset($csrfArr[$k]) && hash_equals($csrfArr[$k], hash_hmac('SHA256', $v, $this->hmacKey))) {
                    if ($deleteToken) {
                        unset($csrfArr[$k]);
                        $this->session->set($this->scrfSessionIndex, $csrfArr);
                    }
                    return true;
                }
            }
            return false;
        }

        /**
         * Generates token for CSRF protection
         *
         * @return array Form index, token and html input tag
         * @throws \Exception
         */
        public function csrfGenerate(): array
        {
            $formInd = bin2hex(random_bytes(15));
            $csrfToken = bin2hex(random_bytes(21));

            $csrfArr = $this->session->get($this->scrfSessionIndex) ?: [];
            $csrfArr[$formInd] = hash_hmac('SHA256', $csrfToken, $this->hmacKey);

            $this->session->set($this->scrfSessionIndex, $csrfArr);

            return [
                'formInd' => $formInd,
                'csrvToken' => $csrfToken,
                'csrfTag' => '<input type="hidden" name="' . $this->encodeSpecialChars($formInd) . '" value="' . $this->encodeSpecialChars($csrfToken) . '">' . "\n"
            ];
        }

        /**
         *
         * @param string $str
         * @return string
         */
        public function cleanXSS(string $str): string
        {
            $str = html_entity_decode($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');

            foreach ($this->xssPreg as $pattern => $rep) {
                $str = preg_replace($pattern, $rep, $str);
            }
            return $str;
        }

        /**
         * Encodes all HTML entities
         *
         * @param string $str
         * @return string
         */
        public function encodeHtmlEntities(string $str): string
        {
            return htmlentities($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }

        /**
         * Encodes special characters
         *
         * @param string $str
         * @return string
         */
        public function encodeSpecialChars(string $str): string
        {
            return htmlspecialchars($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }

    }