/**
 * Check if elem is an HTML element
 * @param {object} elem
 * @return {Boolean}
 */
export function isHtmlElement(elem) {
    return elem && typeof elem === 'object' && elem.nodeType === 1;
}

/**
 * Returns HTML element from ID or node element.
 * Undefined if the element does not exist
 *
 * @param {string|object} elemRef
 * @return {object|undefined}
 */
export function getHtmlElement(elemRef) {
    let elem;
    if (typeof elemRef === 'string') {
        elem = document.getElementById(elemRef);
    } else if (elemRef && typeof elemRef === 'object' && elemRef.nodeType === 1) {
        elem = elemRef;
    }
    return elem;
}

/**
 * Check if a string is numeric
 *
 * @param {string} val
 * @return {Boolean}
 */
export function isNumeric(val) {
    return !isNaN(parseFloat(val)) && isFinite(val);
}

/**
 * Check if a string is "empty"
 *
 * @param {string} val
 * @return {Boolean}
 */
export function isEmpty(val) {
    return !val || (typeof val === 'string' && val.trim() === '');
}

/**
 * Email validation
 *
 * @param {string} str
 * @return {Boolean}
 */
export function validEmail(str) {
    if (typeof str !== "string") {
        return false;
    }

    let val = str.trim(),
        atpos = val.indexOf("@"),
        dotpos = val.lastIndexOf(".");

    if (val.indexOf(" ") !== -1 || atpos < 1 || dotpos < 1) {
        return false;
    } else if (val.indexOf("@", (atpos + 1)) !== -1) {
        return false;
    } else if ((dotpos - atpos) < 2 || (val.length - (dotpos + 1)) < 2) {
        return false;
    } else {
        return true;
    }
}

/**
 * Applies border (warn) to form fields
 *
 * @param {type} elemRef
 * @return {undefined}
 */
export function warnField(elemRef) {
    let elem = getHtmlElement(elemRef);
    if (!elem) {
        return;
    }

    elem.classList.add('warn-field');
}

/**
 * Warning with text message
 *
 * @param {type} elemRef
 * @return {undefined}
 */
export function warnTxt(elemRef) {
    let elem = getHtmlElement(elemRef);
    if (!elem) {
        return;
    }
    elem.classList.add('warn-txt');
}

/**
 * Clear the "warn" styles applied to form fields and text messages
 *
 * @param {type} containerRef
 * @return {undefined}
 */
export function clearWarned(containerRef) {
    let container = getHtmlElement(containerRef);
    if (!container) {
        return;
    }
    let warned = container.getElementsByClassName('warn-field'),
        len = warned.length;
    while (len--) {
        warned[len].classList.remove('warn-field');
    }

    warned = container.getElementsByClassName('warn-txt'),
        len = warned.length;
    while (len--) {
        warned[len].classList.remove('warn-txt');
    }

    warned = container.getElementsByClassName('warnings-list'),
        len = warned.length;
    while (len--) {
        warned[len].parentElement.removeChild(warned[len]);
    }
}

/**
 * Returns the first parent element of the specified class or node name
 *
 * @param {object} elem
 * @param {string} name class name or node(tag) name
 * @return {object|undefined}
 */
export function getParentElement(elem, name) {
    if (!isHtmlElement(elem) || isEmpty(name)) {
        return;
    }

    if (name.slice(0, 1) === '.') { // Class name
        name = name.slice(1);
        while (elem.parentNode) {
            elem = elem.parentNode;
            try {
                if (elem.classList.contains(name)) {
                    return elem;
                }
            } catch (err) {
                return;
            }
        }
    } else { // Tag name
        name = name.toUpperCase();
        while (elem.parentNode) {
            elem = elem.parentNode;
            if (elem.nodeName === name) {
                return elem;
            }
        }
    }
}

/**
 *
 * Append options to select element
 *
 * @param {object|string} elemRef HTML element or id
 * @param {array} opts
 */
export function selectAppendOptions(elemRef, opts) {
    let elem = getHtmlElement(elemRef);

    if (elem && Array.isArray(opts)) {
        const doc = document,
            frag = doc.createDocumentFragment();

        let opt,
            optAttr;

        for (let i = 0, iLen = opts.length; i < iLen; i++) {
            opt = doc.createElement('option');
            if (opts[i].attr) {
                optAttr = opts[i].attr;
                opt.setAttribute(optAttr, optAttr);
            }
            opt.value = opts[i].val;
            opt.text = opts[i].txt;
            frag.appendChild(opt);
        }
        elem.appendChild(frag);
    }
}

/**
 * Remove options from select element, but keep first k options
 *
 * @param {object} elemRef HTML element or id
 * @param {int} k number of options to keep
 */
export function selectRemoveOptions(elemRef, k) {
    let elem = getHtmlElement(elemRef);
    if (elem) {
        let keep = (isNumeric(k) && k >= 0) ? k : 0,
            i = elem.options.length;
        while (i-- > keep) {
            elem.remove(i);
        }
    }
}

export function allowedImgType(fl) {
    let t = [".jpg", ".jpeg", ".gif", ".png"],
        fname = fl.slice(fl.lastIndexOf("\\") + 1) || fl.slice(fl.lastIndexOf("/") + 1),
        ext = fname.toLowerCase().slice(fname.indexOf("."));
    for (let i = 0; i < t.length; i++) {
        if (t[i] === ext) {
            return true;
        }
    }
    return false;
}

export function displayWarnings(containerRef, warnings) {
    let container = getHtmlElement(containerRef);
    clearWarned(containerRef);
    if (container && Array.isArray(warnings)) {
        const doc = document,
            frag = doc.createDocumentFragment();

        let ul,
            li;
        ul = doc.createElement('ul');
        ul.className = "warnings-list";
        for (let i = 0, iLen = warnings.length; i < iLen; i++) {
            li = doc.createElement('li');
            li.textContent = warnings[i];
            frag.appendChild(li);
        }

        ul.appendChild(frag);
        container.appendChild(ul);
    }

}
