<?php

    namespace Asn\Core\Http;

    class Response
    {

        private $code = 200;
        private $headers = [];

        /**
         * Sets HTTP status code
         *
         * @param int $code
         * @return $this
         * @throws \Exception
         */
        public function setCode(int $code)
        {
            if ($code < 100 || $code > 599) {
                throw new \Exception("Invalid status code: $code");
            }
            $this->code = $code;
            return $this;
        }

        /**
         * Returns HTTP status code
         * 
         * @return int
         */
        public function getCode(): int
        {
            return $this->code;
        }

        /**
         * Sets a header. Deletes the header is value is null
         * 
         * @param string $name
         * @param string $value
         * @return $this
         */
        public function setHeader(string $name, string $value = null)
        {
            $name = strtolower(trim($name));
            if ($value === null) {
                if (isset($this->headers[$name])) {
                    unset($this->headers[$name]);
                }
            } else {
                $this->headers[$name][] = $value;
            }
            return $this;
        }

        /**
         * Ads a header
         * 
         * @param string $name
         * @param string $value
         * @return $this
         */
        public function addHeader(string $name, string $value)
        {
            $this->headers[strtolower(trim($name))][] = $value;
            return $this;
        }

        /**
         * Returns a header. Returns all headers if name is null
         * 
         * @param string $name
         * @return array|null
         */
        public function getHeaders(string $name = null)
        {
            if (null !== $name) {
                $name = strtolower(trim($name));
                return isset($this->headers[$name]) ? $this->headers[$name] : null;
            }
            return $this->headers;
        }

        /**
         * Sends the headers
         * 
         * @return $this
         */
        public function sendHeaders()
        {
            foreach ($this->headers as $name => $values) {
                foreach ($values as $v) {
                    header($name . ': ' . $v, false);
                }
            }
            http_response_code($this->code);
            return $this;
        }

        /**
         * Sets content message
         *
         * @param string|int|float|null $content   Anything that can be casted to string
         * @return $this
         */
        public function setContent($content)
        {
            $this->content = (string) $content;
            return $this;
        }

        /**
         * Returns the content message
         * 
         * @return string
         */
        public function getContent(): string
        {
            return $this->content;
        }

        /**
         * Sends the content message
         * 
         * @return $this
         */
        public function sendContent()
        {
            echo $this->content;
            return $this;
        }

        /**
         * Sends the headers and the content message
         * 
         * @return $this
         */
        public function send()
        {
            $this->sendHeaders();
            $this->sendContent();
            return $this;
        }

    }